<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Erlage\Photogram\Tools\CookieManager;
use Erlage\Photogram\Data\Models\Admin\AdminModel;
use Erlage\Photogram\Data\Tables\Admin\AdminTable;
use Erlage\Photogram\Data\Models\Admin\AdminModelHelper;

final class AdminSession
{
    /**
     * @return AdminSession 
     */
    final public static function instance()
    {
        static $instance = false;

        if (false === $instance)
        {
            $instance = new AdminSession();
        }

        return $instance;
    }

    /**
     * @var bool
     */
    protected $isAuthenticated = false;

    /**
     * @var AdminModel
     */
    protected $adminModel;

    public function init(): void
    {
        $adminId = CookieManager::get(AdminTable::TABLE_NAME . '_' . AdminTable::ID);

        if (null != $adminId)
        {
            $adminModel = AdminModel::findFromId_noException($adminId);

            if ($adminModel -> isModel())
            {
                // this will also verify meta access token.
                $this -> setAdminModel($adminModel);

                if ($this -> isAuthenticated())
                {
                    AdminModelHelper::updateLastActive($this -> getAdminModel());
                }
            }
        }
    }

    public function isAuthenticated(): bool
    {
        return $this -> isAuthenticated;
    }

    public function getAdminModel(): AdminModel
    {
        return $this -> adminModel;
    }

    public function loginAs(AdminModel $adminModel): void
    {
        $this -> adminModel = $adminModel;

        $this -> isAuthenticated = true;
    }

    public function setAdminModel(AdminModel $adminModel): void
    {
        $this -> adminModel = $adminModel;

        $this -> verifyMetaAccessToken();
    }

    public function logout()
    {
        $this -> isAuthenticated = false;

        $this -> removeCookies();
    }

    public function updateSession()
    {
        if ($this -> isAuthenticated())
        {
            $this -> addCookies();
        }
    }

    private function verifyMetaAccessToken(): void
    {
        if ($this -> getAdminModel() -> isModel())
        {
            $adminMetaAccess = CookieManager::get(AdminTable::TABLE_NAME . '_' . AdminTable::META_ACCESS);

            if ($adminMetaAccess == $this -> getAdminModel() -> getMetaAccess())
            {
                $this -> isAuthenticated = true;
            }
        }
        else
        {
            $this -> logout();
        }
    }

    private function addCookies()
    {
        CookieManager::set(
            AdminTable::TABLE_NAME . '_' . AdminTable::ID,
            $this -> getAdminModel() -> getId()
        );

        CookieManager::set(
            AdminTable::TABLE_NAME . '_' . AdminTable::META_ACCESS,
            $this -> getAdminModel() -> getMetaAccess()
        );
    }

    private function removeCookies()
    {
        CookieManager::unset(AdminTable::TABLE_NAME . '_' . AdminTable::ID);

        CookieManager::unset(AdminTable::TABLE_NAME . '_' . AdminTable::META_ACCESS);
    }
}
